<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Purchase;
use App\Models\PurchaseDetail;
use App\Models\Member;
use App\Models\Visitor;
use Illuminate\Http\Request;
use Carbon\Carbon;

class PurchaseController extends Controller
{

    public function index(Request $request)
    {
        // Query dasar tanpa relasi
        $query = Purchase::query();
        // Default filter: bulan berjalan
        if (!$request->filled('tanggal_mulai') && !$request->filled('tanggal_selesai')) {
            $tanggalMulai = Carbon::now()->startOfMonth()->toDateString();
            $tanggalSelesai = Carbon::now()->endOfMonth()->toDateString();
            $request->merge([
                'tanggal_mulai' => $tanggalMulai,
                'tanggal_selesai' => $tanggalSelesai,
            ]);
        }

        // Filter tanggal
        if ($request->filled('tanggal_mulai')) {
            $query->whereDate('tanggal', '>=', $request->tanggal_mulai);
        }
        if ($request->filled('tanggal_selesai')) {
            $query->whereDate('tanggal', '<=', $request->tanggal_selesai);
        }
        // Filter pencarian
        if ($request->filled('q')) {
            $search = $request->q;
            $query->where(function ($q) use ($search) {
                $q->whereHas('member', function ($q2) use ($search) {
                    $q2->where('nama', 'like', '%'.$search.'%');
                })
                ->orWhere('nama_pembeli', 'like', '%'.$search.'%')
                ->orWhereHas('details.product', function ($q3) use ($search) {
                    $q3->where('nama', 'like', '%'.$search.'%');
                });
            });
        }

        // Untuk tampilan, baru load relasi
        $purchases = (clone $query)
            ->with(['member', 'details.product'])
            ->orderByDesc('tanggal')
            ->paginate(20)
            ->withQueryString();

        // Untuk total transaksi, cukup sum tanpa relasi
        $totalTransaksi = (clone $query)->sum('total_harga');

        return view('purchases.index', compact('purchases', 'totalTransaksi'));
    }

    public function create()
    {
        //$products = Product::orderBy('nama')->get();
        $products = Product::orderBy('nama')
            ->whereNotIn('kategori', ['pt_sesi', 'pt_visit']) // list kajobo pt
            ->get();
        $members = Member::select('id', 'nama', 'alamat', 'kontak')->get();
        $visitors = Visitor::select('id', 'nama', 'alamat', 'kontak')->get();
        return view('purchases.create', compact('products', 'members', 'visitors'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'buyer_type' => 'required|in:member,visitor',
            'buyer_id' => 'required|integer',
            'payment_method' => 'required|in:cash,QRIS',
            'produk_id' => 'required|array|min:1',
            'qty' => 'required|array',
            'qty.*' => 'required|integer|min:1',
            'bayar' => 'required|integer|min:0',
        ]);

        // Ambil data buyer
        if ($request->buyer_type === 'member') {
            $buyer = Member::findOrFail($request->buyer_id);
        } else {
            $buyer = Visitor::findOrFail($request->buyer_id);
        }

        // Hitung total harga
        $total = 0;
        foreach ($request->produk_id as $i => $pid) {
            $produk = Product::find($pid);
            $total += $produk->harga * $request->qty[$i];
        }

        // Validasi bayar cukup
        if ($request->bayar < $total) {
            return back()->withErrors(['bayar' => 'Nominal bayar kurang dari total belanja.'])->withInput();
        }

        // Simpan purchase
        $purchase = Purchase::create([
            'member_id' => ($request->buyer_type === 'member') ? $buyer->id : null,
            'nama_pembeli' => $buyer->nama,
            'alamat_pembeli' => $buyer->alamat,
            'tanggal' => now(),
            'total_harga' => $total,
            'bayar' => $request->bayar,
            'kembali' => $request->bayar - $total,
            'payment_method' => $request->payment_method,
        ]);

        // Simpan detail
        foreach ($request->produk_id as $i => $pid) {
            $produk = Product::find($pid);
            PurchaseDetail::create([
                'purchase_id' => $purchase->id,
                'product_id' => $pid,
                'qty' => $request->qty[$i],
                'harga_satuan' => $produk->harga,
            ]);
            $produk->decrement('stok', $request->qty[$i]);
        }

        return redirect()->route('purchases.print', $purchase->id);
    }


    public function print(Purchase $purchase)
    {
        // logic cetak (return view/cetak struk/pdf/dll)
        $purchase->load(['member', 'details.product']);
        return view('purchases.print', compact('purchase'));
    }
    
    public function destroy(Purchase $purchase)
    {
        if (auth()->user()->role !== 'admin') {
            abort(403, 'Unauthorized action.');
        } 
        // Hapus detail dulu jika relasi tidak cascade
        $purchase->details()->delete(); // Jika sudah ada relasi details di model Purchase

        $purchase->delete();

        return redirect()->route('purchases.index')
            ->with('success', 'Data pembelian berhasil dihapus.');
    }
    
}