<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    // List produk
    public function index(Request $request)
    {
        $q = $request->query('q');
        $products = Product::when($q, function($query) use ($q) {
            $query->where('nama', 'like', "%$q%")
                  ->orWhere('kategori', 'like', "%$q%");
            })
            ->orderBy('kategori')
            ->orderBy('nama')
            ->paginate(10) // atur jumlah per halaman
            ->appends(['q' => $q]); // agar query string tetap di link pagination

        return view('produk.index', compact('products', 'q'));
    }

    // Form tambah
    public function create()
    {
        $kategori = [
            'air_minum' => 'Air Minum',
            'suplement' => 'Suplement',
            'snack' => 'Snack',
            'merchandise' => 'Merchandise',
            'pt_sesi' => 'PT Sesi',
            'pt_visit' => 'PT Visit'
        ];
        return view('produk.create', compact('kategori'));
    }

    // Simpan produk baru
    public function store(Request $request)
    {
        $request->validate([
            'nama' => 'required',
            'kategori' => 'required|in:air_minum,suplement,snack,merchandise,pt_sesi,pt_visit',
            'stok' => 'required|integer|min:0',
            'harga' => 'required|numeric|min:0',
        ]);
        Product::create($request->only(['nama', 'kategori', 'stok', 'harga']));
        return redirect()->route('produk.index')->with('success', 'Produk berhasil ditambah!');
    }

    // Form edit
    public function edit($id)
    {
        $product = \App\Models\Product::find($id);
        //dd($product); // Debug
        $kategori = [
            'air_minum' => 'Air Minum',
            'suplement' => 'Suplement',
            'snack' => 'Snack',
            'merchandise' => 'Merchandise',
            'pt_sesi' => 'PT Sesi',
            'pt_visit' => 'PT Visit'
        ];
        return view('produk.edit', compact('product', 'kategori'));
    }
    /*public function edit(Product $product)
    {
        $kategori = [
            'air_minum' => 'Air Minum',
            'suplement' => 'Suplement',
            'snack' => 'Snack',
            'merchandise' => 'Merchandise'
        ];

        return view('produk.edit', compact('product', 'kategori'));
    }*/

    // Update produk
    public function update(Request $request, Product $produk)
    {
        $request->validate([
            'nama' => 'required',
            'kategori' => 'required|in:air_minum,suplement,snack,merchandise,pt_sesi,pt_visit',
            'stok' => 'required|integer|min:0',
            'harga' => 'required|numeric|min:0',
        ]);
        $produk->update($request->only(['nama', 'kategori', 'stok', 'harga']));
        return redirect()->route('produk.index')->with('success', 'Produk berhasil diupdate!');
    }

    // Restok
    public function restok(Request $request, Product $produk)
    {
        $request->validate([
            'jumlah' => 'required|integer|min:1'
        ]);
        $produk->stok += $request->jumlah;
        $produk->save();

        return redirect()->back()->with('success', 'Stok produk berhasil ditambah!');
    }
    
    // Hapus produk
    public function destroy(Product $product)
    {
        if (auth()->user()->role !== 'admin') {
            abort(403, 'Unauthorized action.');
        }
        $product->delete();
        return redirect()->route('produk.index')->with('success', 'Produk berhasil dihapus!');
    }
}