<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\StokAlat;
use Illuminate\Support\Facades\Auth;
use App\Models\HistoriAlat;

class StokOpnameController extends Controller
{
    // List alat + filter + pagination
    public function index(Request $request)
    {
        $user = Auth::user();
        // Hanya untuk teknisi-opname, jika ingin view-only role lain, bisa tambahkan pengecekan di sini.
        $canCrud = $user->hasRole('teknisi-opname');

        $query = StokAlat::with('teknisi');
        // Filter nama alat/serial
        if ($request->q) {
            $query->where(function($q) use ($request){
                $q->where('nama_alat', 'like', '%'.$request->q.'%')
                  ->orWhere('serial_number', 'like', '%'.$request->q.'%');
            });
        }
        // Filter tanggal masuk
        if ($request->tanggal_masuk) {
            $query->where('tanggal_masuk', $request->tanggal_masuk);
        }
        // Filter tanggal keluar
        if ($request->tanggal_keluar) {
            $query->where('tanggal_keluar', $request->tanggal_keluar);
        }
        // Filter teknisi
        if ($request->teknisi_id) {
            $query->where('teknisi_id', $request->teknisi_id);
        }

        $alat = $query->orderByDesc('id')->paginate(10)->withQueryString();

        return view('stok_opname.index', compact('alat', 'canCrud'));
    }

    // Simpan alat baru
    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user->hasRole('teknisi-opname')) {
            abort(403);
        }

        $request->validate([
            'nama_alat' => 'required|string|max:100',
            'serial_number' => 'nullable|string|max:100',
            'keluar_untuk' => 'nullable|string|max:255',
            'teknisi_id' => 'nullable|exists:users,id',
            'tanggal_keluar' => 'nullable|date',
            'tanggal_masuk' => 'nullable|date',
            'sumber_masuk' => 'nullable|in:pembelian,pelanggan_berhenti',
            'keterangan' => 'nullable|string',
            'status' => 'required|in:stok,keluar,masuk',
        ]);

        $alat = StokAlat::create($request->all());

        // Catat histori masuk
        HistoriAlat::create([
            'stok_alat_id' => $alat->id,
            'jenis' => 'masuk',
            'sumber_masuk' => $request->sumber_masuk,
            'tanggal' => $request->tanggal_masuk ?? now(),
            'keterangan' => $request->keterangan,
            'created_at' => now(),
        ]);

        return redirect()->route('stok-opname.index')->with('success', 'Alat berhasil ditambahkan!');
    }

    // Update alat (modal)
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        if (!$user->hasRole('teknisi-opname')) {
            abort(403);
        }

        $request->validate([
            'nama_alat' => 'required|string|max:100',
            'serial_number' => 'nullable|string|max:100',
            'keluar_untuk' => 'nullable|string|max:255',
            'teknisi_id' => 'nullable|exists:users,id',
            'tanggal_keluar' => 'nullable|date',
            'tanggal_masuk' => 'nullable|date',
            'sumber_masuk' => 'nullable|in:pembelian,pelanggan_berhenti',
            'keterangan' => 'nullable|string',
            'status' => 'required|in:stok,keluar,masuk',
        ]);

        $alat = StokAlat::findOrFail($id);
        $alat->update($request->all());

        // Catat histori keluar/masuk
        HistoriAlat::create([
            'stok_alat_id' => $alat->id,
            'jenis' => $request->status == 'keluar' ? 'keluar' : 'masuk',
            'keluar_untuk' => $request->keluar_untuk,
            'teknisi_id' => $request->teknisi_id,
            'tanggal' => $request->status == 'keluar' ? $request->tanggal_keluar : $request->tanggal_masuk,
            'sumber_masuk' => $request->sumber_masuk,
            'keterangan' => $request->keterangan,
            'created_at' => now(),
        ]);

        return redirect()->route('stok-opname.index')->with('success', 'Data alat berhasil diupdate!');
    }

    // Hapus alat + histori
    public function destroy($id)
    {
        $user = Auth::user();
        if (!$user->hasRole('teknisi-opname')) {
            abort(403);
        }

        $alat = StokAlat::findOrFail($id);
        // Hapus semua histori terkait (manual, jika belum cascade)
        HistoriAlat::where('stok_alat_id', $alat->id)->delete();
        $alat->delete();

        return redirect()->route('stok-opname.index')->with('success', 'Data alat & histori berhasil dihapus!');
    }
}