<?php

namespace App\Http\Controllers;

use App\Models\Pembayaran;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use App\Models\Pelanggan;
use App\Models\Grub;

class PenagihanController extends Controller
{
    public function index(Request $request)
    {
        $userId = auth()->id();
        $today = \Carbon\Carbon::today();
        $tomorrow = \Carbon\Carbon::tomorrow();
    
        $pembayarans = \App\Models\Pembayaran::select('pembayarans.*')
            ->join('pelanggans', 'pembayarans.pelanggan_id', '=', 'pelanggans.id')
            ->where('pelanggans.penagih_id', $userId)
            ->when($request->filled('grub_id'), function($q) use ($request) {
                $q->where('pelanggans.grub_id', $request->grub_id);
            })
            ->when($request->filled('q'), function($q) use ($request) {
                $q->where(function($sub) use ($request) {
                    $sub->where('pelanggans.nama', 'like', '%' . $request->q . '%')
                        ->orWhere('pelanggans.alamat', 'like', '%' . $request->q . '%')
                        ->orWhere('pelanggans.no_hp', 'like', '%' . $request->q . '%');
                });
            })
            ->where(function($query) use ($today, $tomorrow) {
                // Tagihan hari ini atau besok
                $query->whereRaw(
                    "STR_TO_DATE(CONCAT(pembayarans.periode, '-', DAY(pelanggans.tanggal_berlangganan)), '%Y-%m-%d') = ?", [$today->format('Y-m-d')]
                )
                ->orWhereRaw(
                    "STR_TO_DATE(CONCAT(pembayarans.periode, '-', DAY(pelanggans.tanggal_berlangganan)), '%Y-%m-%d') = ?", [$tomorrow->format('Y-m-d')]
                )
                // Tagihan belum lunas di periode sebelumnya
                ->orWhere(function($q) use ($today) {
                    $q->where('pembayarans.status', 'belum_lunas')
                      ->whereRaw(
                          "STR_TO_DATE(CONCAT(pembayarans.periode, '-', DAY(pelanggans.tanggal_berlangganan)), '%Y-%m-%d') < ?", [$today->format('Y-m-d')]
                      );
                });
            })
            ->with(['pelanggan.grub'])
            ->orderBy('pembayarans.periode', 'desc')
            ->paginate(10);
    
        $grubs = \App\Models\Grub::all();
    
        return view('pembayaran.index', compact('pembayarans', 'grubs'));
    }

    public function bayarSubmit(Request $request, $id)
    {
        $pembayaran = Pembayaran::findOrFail($id);
        $sisa_tagihan = $pembayaran->jumlah_tagihan - $pembayaran->jumlah_dibayar;
    
        $request->validate([
            'jumlah_dibayar' => "required|numeric|min:1|max:$sisa_tagihan",
        ]);
    
        $pembayaran->jumlah_dibayar += $request->jumlah_dibayar;
    
        if ($pembayaran->jumlah_dibayar >= $pembayaran->jumlah_tagihan) {
            $pembayaran->status = 'lunas';
            $pembayaran->jumlah_dibayar = $pembayaran->jumlah_tagihan; // pastikan tidak lebih dari tagihan
        } else {
            $pembayaran->status = 'belum_lunas';
        }
    
        // Tambahkan ini untuk update tanggal pembayaran!
        $pembayaran->tanggal_bayar = now();
    
        $pembayaran->save();
    
        return redirect()->back()->with('success', 'Pembayaran berhasil!');
    }
}