<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\JadwalTeknisi;
use App\Models\JadwalTeknisiUser;
use App\Models\User;
use App\Models\Grub;
use Illuminate\Support\Facades\DB;

class JadwalTeknisiController extends Controller
{
    public function index(Request $request)
    {
        $query = \App\Models\JadwalTeknisi::with(['jadwalTeknisiUsers.teknisi', 'pelanggan']);
    
        // Filter utama: cari di semua kolom
        if ($request->filled('search_q')) {
            $search = $request->search_q;
            $query->where(function($q) use ($search) {
                $q->where('no_tiket', 'like', "%$search%")
                  ->orWhere('lokasi', 'like', "%$search%")
                  ->orWhere('deskripsi_tugas', 'like', "%$search%")
                  ->orWhereHas('jadwalTeknisiUsers.teknisi', function($qq) use ($search) {
                      $qq->where('name', 'like', "%$search%");
                  })
                  ->orWhereHas('pelanggan', function($qq) use ($search) {
                      $qq->where('nama', 'like', "%$search%");
                  });
            });
        }
    
        // Filter tanggal
        if ($request->filled('search_tanggal')) {
            $query->where('tanggal', $request->search_tanggal);
        }
    
        // Filter status
        if ($request->filled('search_status')) {
            $status = $request->search_status;
            if ($status == 'open') {
                $query->whereHas('jadwalTeknisiUsers', function($q){
                    $q->where('status_feedback', 'open');
                });
            } elseif ($status == 'selesai_pending') {
                $query->whereHas('jadwalTeknisiUsers', function($q){
                    $q->where('status_feedback', 'selesai')
                      ->where('status_validasi', 'pending');
                });
            } elseif ($status == 'selesai_validated') {
                $query->whereHas('jadwalTeknisiUsers', function($q){
                    $q->where('status_feedback', 'selesai')
                      ->where('status_validasi', 'validated');
                });
            }
        }
    
        $jadwals = $query->orderBy('tanggal', 'desc')->paginate(10);
    
        return view('jadwal_teknisi.index', compact('jadwals'));
    }


public function create()
{
    $teknisis = User::role('teknisi')->get();
    $pelanggans = \App\Models\Pelanggan::all(); // Tambahkan baris ini!
    return view('jadwal_teknisi.create', compact('teknisis', 'pelanggans'));
}

    public function store(Request $request)
    {
        // Validasi dinamis
        $rules = [
            'teknisi_id' => 'required|array',
            'teknisi_id.*' => 'exists:users,id',
            'jenis_tugas' => 'required|in:mainten infra,mainten pelanggan,pemasangan baru',
            'tanggal' => 'required|date',
            'deskripsi_tugas' => 'required|string',
        ];
    
        if ($request->jenis_tugas == 'mainten infra') {
            $rules['lokasi'] = 'required|string';
        }
        if ($request->jenis_tugas == 'mainten pelanggan' || $request->jenis_tugas == 'pemasangan baru') {
            $rules['pelanggan_id'] = 'required|exists:pelanggans,id';
        }
    
        $request->validate($rules);
    
        // Generate no tiket
        $prefix = $request->jenis_tugas == 'mainten infra' ? 'MIL-' : ($request->jenis_tugas == 'mainten pelanggan' ? 'MPL-' : 'PSBL-');
        $last = \App\Models\JadwalTeknisi::where('jenis_tugas', $request->jenis_tugas)
            ->orderBy('id', 'desc')->first();
        $number = $last ? ((int)preg_replace('/\D/', '', $last->no_tiket) + 1) : 1;
        $no_tiket = $prefix . str_pad($number, 6, '0', STR_PAD_LEFT);
    
        // Hari tugas
        $nama_hari = \Carbon\Carbon::parse($request->tanggal)->locale('id')->isoFormat('dddd');
    
        \DB::beginTransaction();
        try {
            $jadwalData = [
                'no_tiket' => $no_tiket,
                'jenis_tugas' => $request->jenis_tugas,
                'tanggal' => $request->tanggal,
                'nama_hari' => $nama_hari,
                'deskripsi_tugas' => $request->deskripsi_tugas,
                'status' => 'Open',
                'created_by' => auth()->id(),
            ];
    
            if ($request->jenis_tugas == 'mainten infra') {
                $jadwalData['lokasi'] = $request->lokasi;
            }
            if ($request->jenis_tugas == 'mainten pelanggan' || $request->jenis_tugas == 'pemasangan baru') {
                $jadwalData['pelanggan_id'] = $request->pelanggan_id;
            }
    
            $jadwal = \App\Models\JadwalTeknisi::create($jadwalData);
    
            foreach ($request->teknisi_id as $teknisiId) {
                \App\Models\JadwalTeknisiUser::create([
                    'jadwal_teknisi_id' => $jadwal->id,
                    'teknisi_id' => $teknisiId,
                    'status_feedback' => 'open'
                ]);
            }
            \DB::commit();
            return redirect()->route('jadwal-teknisi.index')->with('success', 'Jadwal tugas teknisi berhasil ditambahkan!');
        } catch (\Exception $e) {
            \DB::rollback();
            return back()->with('error', 'Gagal menambah jadwal: '.$e->getMessage());
        }
    }
    
    public function update(Request $request, $id)
    {
        $jadwal = \App\Models\JadwalTeknisi::findOrFail($id);
        // Validasi dinamis
        $rules = [
            'teknisi_id' => 'required|array',
            'teknisi_id.*' => 'exists:users,id',
            'jenis_tugas' => 'required|in:mainten infra,mainten pelanggan,pemasangan baru',
            'tanggal' => 'required|date',
            'deskripsi_tugas' => 'required|string',
        ];
    
        if ($request->jenis_tugas == 'mainten infra') {
            $rules['lokasi'] = 'required|string';
        }
        if ($request->jenis_tugas == 'mainten pelanggan' || $request->jenis_tugas == 'pemasangan baru') {
            $rules['pelanggan_id'] = 'required|exists:pelanggans,id';
        }
    
        $request->validate($rules);
    
        $nama_hari = \Carbon\Carbon::parse($request->tanggal)->locale('id')->isoFormat('dddd');
        \DB::beginTransaction();
        try {
            $jadwalData = [
                'jenis_tugas' => $request->jenis_tugas,
                'tanggal' => $request->tanggal,
                'nama_hari' => $nama_hari,
                'deskripsi_tugas' => $request->deskripsi_tugas,
                // status tidak diubah di sini
            ];
    
            if ($request->jenis_tugas == 'mainten infra') {
                $jadwalData['lokasi'] = $request->lokasi;
                $jadwalData['pelanggan_id'] = null;
            }
            if ($request->jenis_tugas == 'mainten pelanggan' || $request->jenis_tugas == 'pemasangan baru') {
                $jadwalData['pelanggan_id'] = $request->pelanggan_id;
                $jadwalData['lokasi'] = null;
            }
    
            $jadwal->update($jadwalData);
    
            // Update teknisi relasi
            $jadwal->jadwalTeknisiUsers()->delete();
            foreach ($request->teknisi_id as $teknisiId) {
                \App\Models\JadwalTeknisiUser::create([
                    'jadwal_teknisi_id' => $jadwal->id,
                    'teknisi_id' => $teknisiId,
                    'status_feedback' => 'open'
                ]);
            }
            \DB::commit();
            return redirect()->route('jadwal-teknisi.index')->with('success', 'Jadwal tugas teknisi berhasil diupdate!');
        } catch (\Exception $e) {
            \DB::rollback();
            return back()->with('error', 'Gagal update jadwal: '.$e->getMessage());
        }
    }

    public function edit($id)
{
    $jadwal = \App\Models\JadwalTeknisi::with('jadwalTeknisiUsers')->findOrFail($id);
    $teknisis = \App\Models\User::role('teknisi')->get();
    $pelanggans = \App\Models\Pelanggan::all();

    // Ambil id teknisi yang sudah ter-assign
    $jadwal->teknisi_ids = $jadwal->jadwalTeknisiUsers->pluck('teknisi_id')->toArray();

    return view('jadwal_teknisi.edit', [
        'jadwal' => $jadwal,
        'teknisis' => $teknisis,
        'pelanggans' => $pelanggans,
    ]);
}

    public function feedbackForm($jadwalId, $jadwalTeknisiUserId)
    {
        $jadwal = JadwalTeknisi::findOrFail($jadwalId);
        $jadwal_teknisi_user = JadwalTeknisiUser::where('id', $jadwalTeknisiUserId)
            ->where('teknisi_id', auth()->id())->firstOrFail();
        return view('jadwal_teknisi.feedback', compact('jadwal', 'jadwal_teknisi_user'));
    }

    public function feedbackSubmit(Request $request, $jadwalId, $jadwalTeknisiUserId)
    {
        $jadwal = JadwalTeknisi::findOrFail($jadwalId);
        $jadwal_teknisi_user = JadwalTeknisiUser::where('id', $jadwalTeknisiUserId)
            ->where('teknisi_id', auth()->id())->firstOrFail();

        $request->validate([
            'feedback_text' => 'required|string',
            'feedback_photo' => 'nullable|image|max:2048',
        ]);
        /*if ($request->hasFile('feedback_photo')) {
            $path = $request->file('feedback_photo')->store('feedback_photos', 'public');
            $jadwal_teknisi_user->feedback_photo = $path;
        }*/
        if ($request->hasFile('feedback_photo')) {
            $file = $request->file('feedback_photo');
            $filename = uniqid() . '.' . $file->getClientOriginalExtension();
            // Simpan langsung ke public/storage/feedback_photos
            $file->move(public_path('storage/feedback_photos'), $filename);
            $jadwal_teknisi_user->feedback_photo = 'feedback_photos/' . $filename;
        }
        if ($request->filled('nominal_bayar')) {
            $jadwal_teknisi_user->nominal_bayar = $request->nominal_bayar;
            $jadwal_teknisi_user->metode_bayar = $request->metode_bayar;
        } else {
            $jadwal_teknisi_user->nominal_bayar = null;
            $jadwal_teknisi_user->metode_bayar = null;
        }
        $jadwal_teknisi_user->feedback_text = $request->feedback_text;
        $jadwal_teknisi_user->status_feedback = 'selesai';
        $jadwal_teknisi_user->feedback_at = now();
        $jadwal_teknisi_user->save();

        // Cek jika semua teknisi sudah selesai
        $count_all = JadwalTeknisiUser::where('jadwal_teknisi_id', $jadwal->id)->count();
        $count_selesai = JadwalTeknisiUser::where('jadwal_teknisi_id', $jadwal->id)
            ->where('status_feedback', 'selesai')->count();
        if ($count_all > 0 && $count_all == $count_selesai) {
            $jadwal->status = 'Selesai';
            $jadwal->save();
        }

        return redirect()->route('jadwal-saya')->with('success', 'Feedback berhasil dikirim.');
    }

    public function generateTiket(Request $request)
{
    $jenis = $request->jenis_tugas; // 'mainten infra', 'mainten pelanggan', 'pemasangan baru'
    $prefix = $jenis == 'mainten infra' ? 'MIL-' : ($jenis == 'mainten pelanggan' ? 'MPL-' : 'PSBL-');

    // Hitung nomor terakhir berdasarkan jenis
    $last = \App\Models\JadwalTeknisi::where('jenis_tugas', $jenis)
        ->orderBy('id', 'desc')->first();

    $number = $last ? ((int)preg_replace('/\D/', '', $last->no_tiket) + 1) : 1;
    $no_tiket = $prefix . str_pad($number, 6, '0', STR_PAD_LEFT);

    return response()->json(['no_tiket' => $no_tiket]);
}

public function jadwalSaya()
{
    $userId = auth()->id();

    $jadwals = \App\Models\JadwalTeknisi::whereHas('jadwalTeknisiUsers', function($q) use ($userId) {
        $q->where('teknisi_id', $userId);
    })
    ->with(['jadwalTeknisiUsers.teknisi', 'pelanggan'])
    ->get();

    $jadwals = $jadwals->sort(function($a, $b) use ($userId) {
        $a_user = $a->jadwalTeknisiUsers->first();
        $b_user = $b->jadwalTeknisiUsers->first();
        return $this->statusOrder($a_user) <=> $this->statusOrder($b_user);
    });

    // Pagination manual (karena Collection)
    $page = request()->get('page', 1);
    $perPage = 10;
    $jadwals_paginated = new \Illuminate\Pagination\LengthAwarePaginator(
        $jadwals->forPage($page, $perPage),
        $jadwals->count(),
        $perPage,
        $page,
        ['path' => request()->url(), 'query' => request()->query()]
    );

    return view('jadwal_teknisi.jadwal_saya', ['jadwals' => $jadwals_paginated]);
}

private function statusOrder($jadwal_teknisi_user) {
    if (!$jadwal_teknisi_user || $jadwal_teknisi_user->status_feedback != 'selesai') {
        return 0; // Open
    } elseif ($jadwal_teknisi_user->status_feedback == 'selesai' && $jadwal_teknisi_user->status_validasi != 'validated') {
        return 1; // Pending Validasi
    } elseif ($jadwal_teknisi_user->status_feedback == 'selesai' && $jadwal_teknisi_user->status_validasi == 'validated') {
        return 2; // Selesai & Validated
    }
    return 3;
}


public function validasiUpah(Request $request, $id)
{
    $jtuser = JadwalTeknisiUser::findOrFail($id);
    $jtuser->nominal_upah = $request->nominal_upah;
    //$jtuser->keterangan_upah = $request->keterangan_upah;
    $jtuser->status_validasi = 'validated';
    $jtuser->save();
    return back()->with('success', 'Validasi dan upah berhasil disimpan.');
}

public function openFeedback(Request $request, $id)
{
    $jtuser = JadwalTeknisiUser::findOrFail($id);
    $jtuser->status_feedback = 'open';
    $jtuser->status_validasi = 'pending';
    $jtuser->save();
    return back()->with('success', 'Status tugas teknisi dibuka kembali.');
}

}